<?php
/**
 * SellingUnitStock.php 2020-3-20
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */

namespace Gambio\Shop\Stock\ValueObject;

use Gambio\Shop\ProductModifiers\Modifiers\ValueObjects\ModifierIdentifierInterface;
use Gambio\Shop\SellingUnit\Database\Unit\Events\GetSellingUnitReservedQuantityEvent;
use Gambio\Shop\SellingUnit\Database\Unit\Events\Interfaces\GetSellingUnitReservedQuantityEventInterface;
use Gambio\Shop\SellingUnit\Database\Unit\Events\Interfaces\OnGetSellingUnitAvailableQuantityEventInterface;
use Gambio\Shop\SellingUnit\Database\Unit\Events\OnGetSellingUnitAvailableQuantityEvent;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\AvailableQuantity;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Interfaces\QuantityInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\ReservedQuantity;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SelectedQuantity;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitId;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitStockInterface;
use ProductDataInterface;
use Psr\EventDispatcher\EventDispatcherInterface;

class SellingUnitStock implements SellingUnitStockInterface
{

    /**
     * @var EventDispatcherInterface
     */
    protected $dispatcher;

    /**
     * @var SellingUnitId
     */
    protected $id;

    /**
     * @var ProductDataInterface
     */
    protected $product;

    /**
     * @var AvailableQuantity
     */
    protected $availableQuantity;
    
    /**
     * @var ReservedQuantity
     */
    protected $reservedQuantity;

    /**
     * SellingUnitStock constructor.
     * @param SellingUnitId $id
     * @param ProductDataInterface $product
     * @param EventDispatcherInterface $dispatcher
     */
    public function __construct(SellingUnitId $id, ProductDataInterface $product, EventDispatcherInterface $dispatcher)
    {
        $this->dispatcher = $dispatcher;
        $this->id = $id;
        $this->product = $product;
    }

    /**
     * @inheritDoc
     */
    public function availableQuantity(): AvailableQuantity
    {
        if (!$this->availableQuantity) {
            /**
             * @var OnGetSellingUnitAvailableQuantityEventInterface $event
             */
            $event = $this->dispatcher->dispatch(new OnGetSellingUnitAvailableQuantityEvent($this->id, $this->product));
            $this->availableQuantity = $event->quantity();
        }

        return $this->availableQuantity;
    }

    /**
     * @inheritDoc
     */
    public function reservedQuantity(): ReservedQuantity
    {
        if (!$this->reservedQuantity) {
            /**
             * @var GetSellingUnitReservedQuantityEventInterface $event
             */
            $event = $this->dispatcher->dispatch(new GetSellingUnitReservedQuantityEvent($this->id, $this->product));
            $this->reservedQuantity = $event->quantity();
        }

        return $this->reservedQuantity;
    }

    /**
     * @inheritDoc
     */
    public function getQuantityByModifier(ModifierIdentifierInterface $id): QuantityInterface
    {
        $quantity      = $this->availableQuantity()->byModifier($id);
        $reserved      = $this->reservedQuantity()->byModifier($id);
        $quantityValue = ($quantity === null ? 0 : $quantity->value()) - ($reserved === null ? 0 : $reserved->value());
        
        return new SelectedQuantity($quantityValue, '');
    }
}